package cn.gtmap.bdcdj.core.encrypt;

import cn.gtmap.bdcdj.core.encrypt.adapter.decrypt.AnnotationDecryptAdapter;
import cn.gtmap.bdcdj.core.encrypt.adapter.decrypt.CommonDecryptAdapter;
import cn.gtmap.bdcdj.core.encrypt.adapter.decrypt.DecryptAdapter;
import cn.gtmap.bdcdj.core.encrypt.adapter.decrypt.EmptyDecryptAdapter;
import cn.gtmap.bdcdj.core.encrypt.adapter.decrypt.SimpleDecryptAdapter;
import cn.gtmap.bdcdj.core.encrypt.adapter.encrypt.AnnotationEncryptAdapter;
import cn.gtmap.bdcdj.core.encrypt.adapter.encrypt.CommonEncryptAdapter;
import cn.gtmap.bdcdj.core.encrypt.adapter.encrypt.EmptyEncryptAdapter;
import cn.gtmap.bdcdj.core.encrypt.adapter.encrypt.EncryptAdapter;
import cn.gtmap.bdcdj.core.encrypt.adapter.encrypt.SimpleEncryptAdapter;
import cn.gtmap.bdcdj.core.encrypt.annotation.EncryptField;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;

public class EncryptAdapterBuilder {
    private String statementId;
    private Class entityClass;
    private CryptProperties cryptProperties;
    private Object param;
    private static final EncryptAdapter EMPTY_ENCRYPT_ADAPTER = new EmptyEncryptAdapter();
    private static final DecryptAdapter EMPTY_DECRYPT_ADAPTER = new EmptyDecryptAdapter();
    private static final Logger LOGGER = LoggerFactory.getLogger(EncryptAdapterBuilder.class);

    public EncryptAdapterBuilder(String statementId, Class entityClass, Object param, CryptProperties cryptProperties) {
        this.statementId = statementId;
        this.entityClass = entityClass;
        this.param = param;
        this.cryptProperties = cryptProperties;
    }

    public CryptAdapterMatadata build() {
        Method method = this.getMethod();
        CryptAdapterMatadata matadata = new CryptAdapterMatadata();
        matadata.setEncryptAdapter(buildEncryptAdapter(method));
        matadata.setDecryptAdapter(builderDecryptAdapter(method));
        return matadata;
    }

    private EncryptAdapter buildEncryptAdapter(Method method) {
        if (!cryptProperties.isCryptEnable()) {
            return EMPTY_ENCRYPT_ADAPTER;
        } else if (method != null && !ArrayUtils.isEmpty(method.getParameterTypes())) {
            //优先确定是否是commonmapper，这种情况下不会有sql的配置，也不会有注解
            if (method.getDeclaringClass().getName().contains("CommonMapper")) {
                return new CommonEncryptAdapter(method, cryptProperties);
            } else if (hasParamEncryptAnnotation(method)) {//如果参数上有注解，将参数加密
                return new AnnotationEncryptAdapter(cryptProperties);
            } else if(!hasEncryptParam(method)){
                return EMPTY_ENCRYPT_ADAPTER;
            }else{
                return new SimpleEncryptAdapter(cryptProperties);
            }
        } else {
            return EMPTY_ENCRYPT_ADAPTER;
        }
    }

    private DecryptAdapter builderDecryptAdapter(Method method) {
        if (!cryptProperties.isCryptEnable()) {
            return EMPTY_DECRYPT_ADAPTER;
        } else if (method != null && method.getReturnType() != Void.class) {
            if (method.getDeclaringClass().getName().contains("CommonMapper")) {
                return new CommonDecryptAdapter(this.entityClass, cryptProperties);
            }else if(hasMethodEnctyptAnnotation(method)){
                return new AnnotationDecryptAdapter(cryptProperties);
            }else if(!hasDecryptReturn(method)){
                return EMPTY_DECRYPT_ADAPTER;
            }else{
                return new SimpleDecryptAdapter(cryptProperties);
            }
        } else {
            return EMPTY_DECRYPT_ADAPTER;
        }
    }

    private Method getMethod() {
        try {
            Class clazz = Class.forName(this.statementId.substring(0, this.statementId.lastIndexOf(".")));
            String methodName = this.statementId.substring(this.statementId.lastIndexOf(".") + 1);
            Method[] var3 = clazz.getMethods();
            int var4 = var3.length;

            for (int var5 = 0; var5 < var4; ++var5) {
                Method method = var3[var5];
                if (method.getName().equals(methodName)) {
                    return method;
                }
            }

            return null;
        } catch (ClassNotFoundException var7) {
            LOGGER.error("EncryptAdapterBuilder.getMethod error {}", var7);
            return null;
        }
    }

    /**
     * 单一参数可以使用注解进行参数加密
     * @param method
     * @return
     */
    private boolean hasParamEncryptAnnotation(Method method) {
        Class[] parameters = method.getParameterTypes();
        if (ArrayUtils.isEmpty(parameters)) {
            return false;
        } else {
            Annotation[][] parameterAnnotations = method.getParameterAnnotations();
            //注解只对单一参数有效
            if (parameterAnnotations != null && parameterAnnotations.length == 1) {
                Annotation[] annotations = parameterAnnotations[0];
                for (Annotation annotation : annotations) {
                    if (annotation != null && annotation instanceof EncryptField) {
                        return true;
                    }
                }
            }
            if(CollectionUtils.isNotEmpty(cryptProperties.getSqlSingleEncryptFields())){
                for (String field : cryptProperties.getSqlSingleEncryptFields()) {
                    String name = method.getDeclaringClass().getName()+"."+method.getName();
                    if(StringUtils.isNotBlank(name) && name.endsWith(field)){
                        return true;
                    }
                }
            }
            return false;
        }
    }

    /**
     * 当参数只有一个，且为基本类型时，认为不需要加加密（此时需要加加密应当使用注解）
     * @param method
     * @return
     */
    private boolean hasEncryptParam(Method method){
        Class<?>[] parameterTypes = method.getParameterTypes();
        if(parameterTypes != null && parameterTypes.length == 1 && parameterTypes[0] == String.class){
            return false;//不需要加密 返回false
        }
        return true;//需要加密 返回true
    }

    /**
     * 当返回值时string时，不能进行解密，（需要加密要添加注解）
     * @param method
     * @return
     */
    private boolean hasDecryptReturn(Method method){
        Class<?> returnType = method.getReturnType();
        if(returnType != null && returnType == String.class){
            return false;//不需要解密，返回false
        }else{
            return true;//需要解密，返回true
        }
    }

    private boolean hasMethodEnctyptAnnotation(Method method){
        EncryptField annotation = method.getAnnotation(EncryptField.class);
        if(annotation != null){
            return true;
        }
        if(CollectionUtils.isNotEmpty(cryptProperties.getSqlSingleDecryptFields())){
            for (String field : cryptProperties.getSqlSingleDecryptFields()) {
                String name = method.getDeclaringClass().getName()+"."+method.getName();
                if(StringUtils.isNotBlank(name) && name.endsWith(field)){
                    return true;
                }
            }
        }
        return false;
    }
}
