package cn.gtmap.bdcdj.core.encrypt.utils;

import cn.gtmap.bdcdj.core.encrypt.CryptProperties;
import cn.gtmap.bdcdj.core.encrypt.enums.SaltsEnum;
import cn.gtmap.bdcdj.core.encrypt.executor.CryptExecutorFactory;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.pqc.math.linearalgebra.ByteUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.KeyGenerator;
import javax.crypto.spec.SecretKeySpec;
import java.security.Key;
import java.security.SecureRandom;
import java.security.Security;
import java.util.Arrays;

public class GmSm4Util {
    private static final String ENCODING = "UTF-8";
    public static final String ALGORITHM_NAME = "SM4";
    public static final String ALGORITHM_NAME_ECB_PADDING = "SM4/ECB/PKCS5Padding";
    public static final int DEFAULT_KEY_SIZE = 128;

    private static final Logger LOGGER = LoggerFactory.getLogger(GmSm4Util.class);
    public GmSm4Util() {
    }

    private static Cipher generateEcbCipher(String algorithmName, int mode, byte[] key) throws Exception {
        Cipher cipher = Cipher.getInstance(algorithmName, "BC");
        Key sm4Key = new SecretKeySpec(key, "SM4");
        cipher.init(mode, sm4Key);
        return cipher;
    }

    public static byte[] generateKey() throws Exception {
        return generateKey(128);
    }

    public static byte[] generateKey(int keySize) throws Exception {
        KeyGenerator kg = KeyGenerator.getInstance("SM4", "BC");
        kg.init(keySize, new SecureRandom());
        return kg.generateKey().getEncoded();
    }

    public static String encryptEcb(String hexKey, String paramStr) throws Exception {
        String cipherText = "";
        byte[] keyData = ByteUtils.fromHexString(hexKey);
        byte[] srcData = paramStr.getBytes("UTF-8");
        byte[] cipherArray = encrypt_Ecb_Padding(keyData, srcData);
        cipherText = ByteUtils.toHexString(cipherArray);
        return cipherText;
    }

    public static String encryptWithSalt(String hexKey, String paramStr) {
        if (StringUtils.startsWithIgnoreCase(paramStr, SaltsEnum.GM_SM4_SALTS.getSalt())) {
            return paramStr;
        } else {
            byte[] keyData = ByteUtils.fromHexString(hexKey);

            try {
                byte[] srcData = paramStr.getBytes("UTF-8");
                byte[] cipherArray = encrypt_Ecb_Padding(keyData, srcData);
                return SaltsEnum.GM_SM4_SALTS.getSalt() + ByteUtils.toHexString(cipherArray);
            } catch (Exception var5) {
                throw new RuntimeException(var5);
            }
        }
    }

    public static String encryptWithUserSalt(String hexKey, String paramStr, String salt, String encoding) {
        if (StringUtils.isNotEmpty(salt) && StringUtils.startsWithIgnoreCase(paramStr, salt)) {
            return paramStr;
        } else {
            if (hexKey.length() == 16) {
                hexKey = toHexString(hexKey);
            }

            byte[] keyData = ByteUtils.fromHexString(hexKey);

            try {
                byte[] srcData = paramStr.getBytes(encoding);
                byte[] cipherArray = encrypt_Ecb_Padding(keyData, srcData);
                return salt + ByteUtils.toHexString(cipherArray);
            } catch (Exception var8) {
                throw new RuntimeException(var8);
            }
        }
    }

    public static String toHexString(String s) {
        String str = "";

        for(int i = 0; i < s.length(); ++i) {
            int ch = s.charAt(i);
            String s4 = Integer.toHexString(ch);
            str = str + s4;
        }

        return str;
    }

    public static String encryptEcb(String paramStr) {
        if (StringUtils.startsWithIgnoreCase(paramStr, SaltsEnum.GM_SM4_SALTS.getSalt())) {
            return paramStr;
        } else {
            byte[] keyData = ByteUtils.fromHexString("645B0A010D939BD614A620BCAF645470");

            try {
                byte[] srcData = paramStr.getBytes("UTF-8");
                byte[] cipherArray = encrypt_Ecb_Padding(keyData, srcData);
                return SaltsEnum.GM_SM4_SALTS.getSalt() + ByteUtils.toHexString(cipherArray);
            } catch (Exception var4) {
                return paramStr;
            }
        }
    }

    public static byte[] encrypt_Ecb_Padding(byte[] key, byte[] data) throws Exception {
        Cipher cipher = generateEcbCipher("SM4/ECB/PKCS5Padding", 1, key);
        return cipher.doFinal(data);
    }

    public static String decryptEcb(String hexKey, String cipherText) throws Exception {
        String decryptStr = "";
        byte[] keyData = ByteUtils.fromHexString(hexKey);
        byte[] cipherData = ByteUtils.fromHexString(cipherText);
        byte[] srcData = decrypt_Ecb_Padding(keyData, cipherData);
        decryptStr = new String(srcData, "UTF-8");
        return decryptStr;
    }

    public static String decryptWithSalt(String hexKey, String cipherText) throws Exception {
        byte[] keyData = ByteUtils.fromHexString(hexKey);
        byte[] cipherData;
        byte[] srcData;
        if (StringUtils.startsWithIgnoreCase(cipherText, SaltsEnum.GM_SM4_SALTS.getSalt())) {
            if (StringUtils.endsWith(cipherText, "等")) {
                String realCipherText = StringUtils.removeEnd(StringUtils.removeStartIgnoreCase(cipherText, SaltsEnum.GM_SM4_SALTS.getSalt()), "等");
                srcData = ByteUtils.fromHexString(realCipherText);
                srcData = decrypt_Ecb_Padding(keyData, srcData);
                return new String(srcData, "UTF-8") + "等";
            } else {
                cipherData = ByteUtils.fromHexString(StringUtils.removeStartIgnoreCase(cipherText, SaltsEnum.GM_SM4_SALTS.getSalt()));
                srcData = decrypt_Ecb_Padding(keyData, cipherData);
                return new String(srcData, "UTF-8");
            }
        } else {
            cipherData = ByteUtils.fromHexString(cipherText);
            srcData = decrypt_Ecb_Padding(keyData, cipherData);
            return new String(srcData, "UTF-8");
        }
    }

    public static String decryptWithUserSalt(String hexKey, String cipherText, String salt, String encoding) throws Exception {
        if (hexKey.length() == 16) {
            hexKey = toHexString(hexKey);
        }

        byte[] keyData = ByteUtils.fromHexString(hexKey);
        byte[] cipherData;
        byte[] srcData;
        if (StringUtils.startsWithIgnoreCase(cipherText, salt)) {
            if (StringUtils.endsWith(cipherText, "等")) {
                String realCipherText = StringUtils.removeEnd(StringUtils.removeStartIgnoreCase(cipherText, salt), "等");
                srcData = ByteUtils.fromHexString(realCipherText);
                srcData = decrypt_Ecb_Padding(keyData, srcData);
                return new String(srcData, encoding) + "等";
            } else {
                cipherData = ByteUtils.fromHexString(StringUtils.removeStartIgnoreCase(cipherText, salt));
                srcData = decrypt_Ecb_Padding(keyData, cipherData);
                return new String(srcData, encoding);
            }
        } else {
            cipherData = ByteUtils.fromHexString(cipherText);
            srcData = decrypt_Ecb_Padding(keyData, cipherData);
            return new String(srcData, encoding);
        }
    }

    public static String decryptEcb(String cipherText) throws Exception {
        byte[] keyData = ByteUtils.fromHexString("645B0A010D939BD614A620BCAF645470");
        byte[] cipherData;
        byte[] srcData;
        if (StringUtils.startsWithIgnoreCase(cipherText, SaltsEnum.GM_SM4_SALTS.getSalt())) {
            if (StringUtils.endsWith(cipherText, "等")) {
                String realCipherText = StringUtils.removeEnd(StringUtils.removeStartIgnoreCase(cipherText, SaltsEnum.GM_SM4_SALTS.getSalt()), "等");
                srcData = ByteUtils.fromHexString(realCipherText);
                srcData = decrypt_Ecb_Padding(keyData, srcData);
                return new String(srcData, "UTF-8") + "等";
            } else {
                cipherData = ByteUtils.fromHexString(StringUtils.removeStartIgnoreCase(cipherText, SaltsEnum.GM_SM4_SALTS.getSalt()));
                srcData = decrypt_Ecb_Padding(keyData, cipherData);
                return new String(srcData, "UTF-8");
            }
        } else {
            cipherData = ByteUtils.fromHexString(cipherText);
            srcData = decrypt_Ecb_Padding(keyData, cipherData);
            return new String(srcData, "UTF-8");
        }
    }

    public static byte[] decrypt_Ecb_Padding(byte[] key, byte[] cipherText) throws Exception {
        Cipher cipher = null;
        try {
            cipher = generateEcbCipher("SM4/ECB/PKCS5Padding", 2, key);
        } catch (Exception e) {
            e.printStackTrace();
            return "加密字段错误，请联系管理员检查，error:0".getBytes("UTF-8");
        }
        if(cipher != null) {
            try {
                return cipher.doFinal(cipherText);
            } catch (IllegalBlockSizeException e) {
                LOGGER.error("解密失败，IllegalBlockSizeException");
                throw e;
            } catch (BadPaddingException e) {
                LOGGER.error("解密失败，BadPaddingException");
                throw e;
            }
        }else{
            LOGGER.info("解密失败");
            return "加密字段错误，请联系管理员检查，error:3".getBytes("UTF-8");
        }
    }

    public static boolean verifyEcb(String hexKey, String cipherText, String paramStr) throws Exception {
        boolean flag = false;
        byte[] keyData = ByteUtils.fromHexString(hexKey);
        byte[] cipherData = ByteUtils.fromHexString(cipherText);
        byte[] decryptData = decrypt_Ecb_Padding(keyData, cipherData);
        byte[] srcData = paramStr.getBytes("UTF-8");
        flag = Arrays.equals(decryptData, srcData);
        return flag;
    }

    public static void main(String[] args) {
        try {
            String hexKey = ByteUtils.toHexString(generateKey());
            System.out.println("生成秘钥：" + hexKey);
            System.out.println("开始测试SM4加密解密====================");
            String json = "3400000111908050000005";
            System.out.println("加密前：" + json);
            String cipher = encryptEcb(json);
            System.out.println("加密后：" + cipher);
            System.out.println("加密后长度：" + cipher.length());
            json = decryptEcb(cipher);
            System.out.println("解密后：" + json);
            System.out.println("结束===================");
        } catch (Exception var4) {
            var4.printStackTrace();
        }

    }

    public static String encryptEcb(String paramStr, CryptProperties cryptProperties) {
        if (!StringUtils.isBlank(paramStr) && (!StringUtils.isNotEmpty(cryptProperties.getSm4Salt()) || !StringUtils.startsWithIgnoreCase(paramStr, cryptProperties.getSm4Salt()))) {
            return cryptProperties.isUserSm4KeyEnable() ? encryptWithUserSalt(cryptProperties.getSm4key(), paramStr, cryptProperties.getSm4Salt(), cryptProperties.getEncoding()) : encryptEcb(paramStr);
        } else {
            return paramStr;
        }
    }

    public static String decryptEcb(String cipherText, CryptProperties cryptProperties) throws Exception {
        if (StringUtils.isBlank(cipherText)) {
            return cipherText;
        } else {
            return cryptProperties.isUserSm4KeyEnable() ? decryptWithUserSalt(cryptProperties.getSm4key(), cipherText, cryptProperties.getSm4Salt(), cryptProperties.getEncoding()) : decryptEcb(cipherText);
        }
    }

    static {
        Security.addProvider(new BouncyCastleProvider());
    }
}
